<?php
/**
 * WooCommerce Integration
 *
 * Protects WooCommerce checkout and product reviews from spam
 */
class Spam_Analyzer_WooCommerce
{
    /**
     * Validate WooCommerce checkout before order placement
     *
     * @param array $data Posted checkout data
     * @param WP_Error $errors Validation errors object
     */
    public function validate_checkout($data, $errors)
    {
        if (!get_option('spam_analyzer_enable_woocommerce')) {
            return;
        }

        // Skip if user is logged in and has previous orders (trusted customer)
        if (is_user_logged_in()) {
            $customer = new WC_Customer(get_current_user_id());
            if ($customer->get_order_count() > 0) {
                return; // Existing customer with order history - skip validation
            }
        }

        // Extract customer data
        $email = isset($data['billing_email']) ? sanitize_email($data['billing_email']) : '';

        if (empty($email)) {
            return; // No email, cannot validate
        }

        // Build validation body with all checkout fields
        $body = array(
            'name' => trim(($data['billing_first_name'] ?? '') . ' ' . ($data['billing_last_name'] ?? '')),
            'phone' => $data['billing_phone'] ?? '',
            'company' => $data['billing_company'] ?? '',
            'address' => trim(($data['billing_address_1'] ?? '') . ' ' . ($data['billing_address_2'] ?? '')),
            'city' => $data['billing_city'] ?? '',
            'state' => $data['billing_state'] ?? '',
            'postcode' => $data['billing_postcode'] ?? '',
            'country' => $data['billing_country'] ?? '',
            'order_notes' => $data['order_comments'] ?? '',
        );

        // Remove empty fields
        $body = array_filter($body, function($value) {
            return !empty(trim($value));
        });

        // Get cart details for context
        $cart = WC()->cart;
        $cart_total = $cart ? $cart->get_total('') : 0;
        $item_count = $cart ? $cart->get_cart_contents_count() : 0;

        // Build context with cart info
        $context = sprintf(
            'WooCommerce checkout on e-commerce website. Cart: %d items, Total: %s',
            $item_count,
            wc_price($cart_total)
        );

        // Validate with API
        $validation_result = Spam_Analyzer_Validator::validate(
            array(
                'context' => $context,
                'email' => $email,
                'body' => $body,
            ),
            'woocommerce_checkout',
            null,
            $email
        );

        // Block if spam detected
        if (Spam_Analyzer_Validator::should_block($validation_result)) {
            $errors->add(
                'spam_detected',
                __('We were unable to process your order. Please contact us directly if you believe this is an error.', 'spam-analyzer')
            );
        }
    }

    /**
     * Validate WooCommerce product reviews
     *
     * @param int $comment_approved Approval status
     * @param array $commentdata Comment data
     * @return int|string Modified approval status
     */
    public function validate_review($comment_approved, $commentdata)
    {
        if (!get_option('spam_analyzer_enable_woocommerce_reviews')) {
            return $comment_approved;
        }

        // Only validate product reviews (comment_type = 'review')
        if (!isset($commentdata['comment_type']) || $commentdata['comment_type'] !== 'review') {
            return $comment_approved;
        }

        // Skip validation for logged-in users with previous reviews
        if (is_user_logged_in()) {
            $user_id = get_current_user_id();
            $previous_reviews = get_comments(array(
                'user_id' => $user_id,
                'type' => 'review',
                'status' => 'approve',
                'count' => true,
            ));

            if ($previous_reviews > 0) {
                return $comment_approved; // User has approved reviews - skip validation
            }
        }

        $email = $commentdata['comment_author_email'];

        if (empty($email)) {
            return $comment_approved;
        }

        // Get product info for context
        $product_id = isset($commentdata['comment_post_ID']) ? $commentdata['comment_post_ID'] : 0;
        $product = $product_id ? wc_get_product($product_id) : null;
        $product_name = $product ? $product->get_name() : 'Unknown Product';

        // Build validation body
        $body = array(
            'name' => $commentdata['comment_author'],
            'message' => $commentdata['comment_content'],
            'product' => $product_name,
            'rating' => isset($commentdata['rating']) ? $commentdata['rating'] : null,
        );

        // Remove empty fields
        $body = array_filter($body);

        // Validate with API
        $validation_result = Spam_Analyzer_Validator::validate(
            array(
                'context' => sprintf('Product review for "%s" on WooCommerce store', $product_name),
                'email' => $email,
                'body' => $body,
            ),
            'woocommerce_review',
            $product_id,
            $email
        );

        // If spam, mark for moderation or trash
        if (Spam_Analyzer_Validator::should_block($validation_result)) {
            $block_mode = get_option('spam_analyzer_block_mode', 'block');

            if ($block_mode === 'block') {
                return 'spam'; // Mark as spam (will be rejected)
            } else {
                return 'hold'; // Hold for moderation
            }
        }

        return $comment_approved;
    }

    /**
     * Prevent fake/spam account registrations during checkout
     *
     * @param WP_Error $errors Validation errors
     * @param string $username Username
     * @param string $email Email address
     */
    public function validate_registration($errors, $username, $email)
    {
        if (!get_option('spam_analyzer_enable_woocommerce_registration')) {
            return;
        }

        if (empty($email)) {
            return;
        }

        // Build validation body
        $body = array(
            'username' => $username,
            'registration_type' => 'woocommerce_checkout',
        );

        // Validate with API
        $validation_result = Spam_Analyzer_Validator::validate(
            array(
                'context' => 'New account registration during WooCommerce checkout',
                'email' => $email,
                'body' => $body,
            ),
            'woocommerce_registration',
            null,
            $email
        );

        // Block if spam detected
        if (Spam_Analyzer_Validator::should_block($validation_result)) {
            $errors->add(
                'spam_registration',
                __('Unable to create account. Please contact us if you need assistance.', 'spam-analyzer')
            );
        }
    }
}
