<?php
/**
 * The core plugin class
 *
 * @package    Spam_Analyzer
 * @subpackage Spam_Analyzer/includes
 */

class Spam_Analyzer
{
    /**
     * The loader that's responsible for maintaining and registering all hooks
     */
    protected $loader;

    /**
     * The unique identifier of this plugin
     */
    protected $plugin_name;

    /**
     * The current version of the plugin
     */
    protected $version;

    /**
     * Initialize the class and set its properties
     */
    public function __construct()
    {
        $this->version = SPAM_ANALYZER_VERSION;
        $this->plugin_name = 'spam-analyzer';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->define_form_integrations();
    }

    /**
     * Load the required dependencies for this plugin
     */
    private function load_dependencies()
    {
        // Core classes
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/class-spam-analyzer-loader.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/class-spam-analyzer-i18n.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/class-spam-analyzer-api.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/class-spam-analyzer-validator.php';

        // Admin classes
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'admin/class-spam-analyzer-admin.php';

        // Form integrations
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-contact-form-7.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-gravity-forms.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-wpforms.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-comments.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-registration.php';
        require_once SPAM_ANALYZER_PLUGIN_DIR . 'includes/integrations/class-spam-analyzer-woocommerce.php';

        $this->loader = new Spam_Analyzer_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization
     */
    private function set_locale()
    {
        $plugin_i18n = new Spam_Analyzer_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     */
    private function define_admin_hooks()
    {
        $plugin_admin = new Spam_Analyzer_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');
        $this->loader->add_action('wp_dashboard_setup', $plugin_admin, 'add_dashboard_widget');
        $this->loader->add_filter('plugin_action_links_' . SPAM_ANALYZER_BASENAME, $plugin_admin, 'add_action_links');
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     */
    private function define_public_hooks()
    {
        // Public hooks if needed (currently all validation is backend)
    }

    /**
     * Register form integration hooks
     */
    private function define_form_integrations()
    {
        // Contact Form 7
        if (class_exists('WPCF7')) {
            $cf7 = new Spam_Analyzer_Contact_Form_7();
            $this->loader->add_filter('wpcf7_validate', $cf7, 'validate_submission', 20, 2);
        }

        // Gravity Forms
        if (class_exists('GFForms')) {
            $gf = new Spam_Analyzer_Gravity_Forms();
            $this->loader->add_filter('gform_validation', $gf, 'validate_submission');
        }

        // WPForms
        if (function_exists('wpforms')) {
            $wpf = new Spam_Analyzer_WPForms();
            $this->loader->add_filter('wpforms_process_before_form_data', $wpf, 'validate_submission', 10, 2);
        }

        // WordPress Comments
        $comments = new Spam_Analyzer_Comments();
        $this->loader->add_filter('preprocess_comment', $comments, 'validate_comment');

        // User Registration
        $registration = new Spam_Analyzer_Registration();
        $this->loader->add_filter('registration_errors', $registration, 'validate_registration', 10, 3);

        // WooCommerce
        if (class_exists('WooCommerce')) {
            $woocommerce = new Spam_Analyzer_WooCommerce();
            // Checkout validation
            $this->loader->add_action('woocommerce_after_checkout_validation', $woocommerce, 'validate_checkout', 10, 2);
            // Product review validation
            $this->loader->add_filter('preprocess_comment', $woocommerce, 'validate_review', 10, 2);
            // Registration during checkout
            $this->loader->add_action('woocommerce_register_post', $woocommerce, 'validate_registration', 10, 3);
        }
    }

    /**
     * Run the loader to execute all of the hooks with WordPress
     */
    public function run()
    {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it
     */
    public function get_plugin_name()
    {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin
     */
    public function get_loader()
    {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin
     */
    public function get_version()
    {
        return $this->version;
    }
}
