<?php
/**
 * Validation handler
 */
class Spam_Analyzer_Validator
{
    /**
     * Validate content with SpamKiller API
     */
    public static function validate($data, $type, $entity_id = null, $email = '')
    {
        $api_key = get_option('spam_analyzer_api_key');

        if (empty($api_key)) {
            return null; // No API key configured, allow submission
        }

        try {
            $api = new Spam_Analyzer_API($api_key);
            $result = $api->validate($data);

            if (!isset($result['data'])) {
                return null;
            }

            $validation_data = $result['data'];

            // Log validation
            self::log_validation(
                $type,
                $entity_id,
                $validation_data['is_spam'],
                $validation_data['spam_score'],
                $validation_data['confidence'],
                self::get_client_ip(),
                $email,
                false,
                wp_json_encode($validation_data)
            );

            // Log to WordPress if enabled
            if (get_option('spam_analyzer_log_activity')) {
                $status_text = $validation_data['is_spam'] ? 'SPAM' : 'LEGITIMATE';
                error_log(sprintf(
                    'SpamKiller: %s - %s (Score: %.1f%%, Confidence: %.1f%%)',
                    $type,
                    $status_text,
                    $validation_data['spam_score'],
                    $validation_data['confidence']
                ));
            }

            return $validation_data;

        } catch (Exception $e) {
            error_log('SpamKiller Error: ' . $e->getMessage());
            return null; // Allow submission on error
        }
    }

    /**
     * Check if content should be blocked
     */
    public static function should_block($result)
    {
        if (!$result) {
            return false;
        }

        $threshold = (float) get_option('spam_analyzer_threshold', 70);
        $block_mode = get_option('spam_analyzer_block_mode', 'block');

        return $result['is_spam']
            && $result['spam_score'] >= $threshold
            && $block_mode === 'block';
    }

    /**
     * Log validation to database
     */
    private static function log_validation($type, $entity_id, $is_spam, $spam_score, $confidence, $ip_address, $email, $blocked, $validation_data)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'spam_analyzer_logs';

        $wpdb->insert(
            $table_name,
            array(
                'type' => $type,
                'entity_id' => $entity_id,
                'is_spam' => $is_spam ? 1 : 0,
                'spam_score' => $spam_score,
                'confidence' => $confidence,
                'ip_address' => $ip_address,
                'email' => $email,
                'blocked' => $blocked ? 1 : 0,
                'validation_data' => $validation_data,
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%d', '%d', '%f', '%f', '%s', '%s', '%d', '%s', '%s')
        );
    }

    /**
     * Get client IP address
     */
    private static function get_client_ip()
    {
        $ip = '';

        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }

        return sanitize_text_field($ip);
    }

    /**
     * Get statistics
     */
    public static function get_statistics($days = 30)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'spam_analyzer_logs';
        $since_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        $stats = $wpdb->get_row($wpdb->prepare("
            SELECT
                COUNT(*) as total,
                SUM(CASE WHEN is_spam = 1 THEN 1 ELSE 0 END) as spam,
                SUM(CASE WHEN is_spam = 0 THEN 1 ELSE 0 END) as legitimate,
                SUM(CASE WHEN blocked = 1 THEN 1 ELSE 0 END) as blocked
            FROM {$table_name}
            WHERE created_at >= %s
        ", $since_date), ARRAY_A);

        return array(
            'total' => (int) $stats['total'],
            'spam' => (int) $stats['spam'],
            'legitimate' => (int) $stats['legitimate'],
            'blocked' => (int) $stats['blocked'],
        );
    }
}
