<?php
/**
 * SpamKiller API Client
 */
class Spam_Analyzer_API
{
    const API_BASE_URL = 'https://spamkiller.io/api/v1';

    private $api_key;

    public function __construct($api_key = null)
    {
        $this->api_key = $api_key ?: get_option('spam_analyzer_api_key');
    }

    /**
     * Get subscription status
     */
    public function get_status()
    {
        return $this->request('GET', '/status');
    }

    /**
     * Validate content for spam
     *
     * @param array $data Must contain 'context', 'email' (required), and 'body' (object)
     * @return array API response
     * @throws Exception If email is missing or body contains 'email' key
     */
    public function validate($data)
    {
        // Email is required
        if (empty($data['email'])) {
            throw new Exception('Email is required for spam validation');
        }

        // Body must be provided
        if (empty($data['body']) || !is_array($data['body'])) {
            throw new Exception('Body must be a non-empty array');
        }

        // Body cannot contain 'email' key
        if (isset($data['body']['email'])) {
            throw new Exception('Body cannot contain "email" key. Email must be provided at top level.');
        }

        $payload = array(
            'context' => isset($data['context']) ? $data['context'] : 'Contact form submission',
            'email' => $data['email'],
            'body' => $data['body'],  // Body is now an object (associative array in PHP)
        );

        return $this->request('POST', '/validate', $payload);
    }

    /**
     * Make HTTP request to API
     */
    private function request($method, $endpoint, $data = array())
    {
        $url = self::API_BASE_URL . $endpoint;

        $args = array(
            'method'  => $method,
            'timeout' => 10,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
                'User-Agent'    => 'WordPress-SpamAnalyzer/' . SPAM_ANALYZER_VERSION,
            ),
        );

        if ($method === 'POST') {
            $args['body'] = wp_json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new Exception('API Request Failed: ' . $response->get_error_message());
        }

        $http_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);

        if ($http_code !== 200) {
            $error_message = isset($result['message']) ? $result['message'] : 'Unknown error';
            throw new Exception('API Error (' . $http_code . '): ' . $error_message);
        }

        if (!isset($result['success']) || !$result['success']) {
            throw new Exception('API returned unsuccessful response');
        }

        return $result;
    }

    /**
     * Check if API key is valid
     */
    public function is_valid_api_key()
    {
        try {
            $this->get_status();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}
