<?php
/**
 * The admin-specific functionality of the plugin
 */
class Spam_Analyzer_Admin
{
    private $plugin_name;
    private $version;

    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area
     */
    public function enqueue_styles()
    {
        wp_enqueue_style($this->plugin_name, SPAM_ANALYZER_PLUGIN_URL . 'assets/css/spam-analyzer-admin.css', array(), $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area
     */
    public function enqueue_scripts()
    {
        wp_enqueue_script($this->plugin_name, SPAM_ANALYZER_PLUGIN_URL . 'assets/js/spam-analyzer-admin.js', array('jquery'), $this->version, false);
    }

    /**
     * Add plugin admin menu
     */
    public function add_plugin_admin_menu()
    {
        add_menu_page(
            __('SpamKiller', 'spam-analyzer'),
            __('SpamKiller', 'spam-analyzer'),
            'manage_options',
            $this->plugin_name,
            array($this, 'display_plugin_admin_page'),
            'dashicons-shield-alt',
            80
        );
    }

    /**
     * Display the plugin admin page
     */
    public function display_plugin_admin_page()
    {
        include_once SPAM_ANALYZER_PLUGIN_DIR . 'admin/partials/spam-analyzer-admin-display.php';
    }

    /**
     * Register plugin settings
     */
    public function register_settings()
    {
        register_setting('spam_analyzer_options', 'spam_analyzer_api_key', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('spam_analyzer_options', 'spam_analyzer_threshold', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_cf7', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_gf', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_wpforms', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_comments', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_registration', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_woocommerce', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_woocommerce_reviews', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_enable_woocommerce_registration', array('sanitize_callback' => 'absint'));
        register_setting('spam_analyzer_options', 'spam_analyzer_block_mode', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('spam_analyzer_options', 'spam_analyzer_log_activity', array('sanitize_callback' => 'absint'));
    }

    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget()
    {
        wp_add_dashboard_widget(
            'spam_analyzer_dashboard_widget',
            __('SpamKiller Statistics', 'spam-analyzer'),
            array($this, 'render_dashboard_widget')
        );
    }

    /**
     * Render dashboard widget
     */
    public function render_dashboard_widget()
    {
        $stats = Spam_Analyzer_Validator::get_statistics(30);
        $api_key = get_option('spam_analyzer_api_key');

        if (empty($api_key)) {
            echo '<p>' . __('Please configure your API key to start using SpamKiller.', 'spam-analyzer') . '</p>';
            echo '<p><a href="' . admin_url('admin.php?page=spam-analyzer') . '" class="button button-primary">' . __('Configure Now', 'spam-analyzer') . '</a></p>';
            return;
        }

        ?>
        <div class="spam-analyzer-widget">
            <div class="stats-grid">
                <div class="stat-box">
                    <div class="stat-number"><?php echo number_format($stats['total']); ?></div>
                    <div class="stat-label"><?php _e('Total Validations', 'spam-analyzer'); ?></div>
                </div>
                <div class="stat-box stat-danger">
                    <div class="stat-number"><?php echo number_format($stats['spam']); ?></div>
                    <div class="stat-label"><?php _e('Spam Detected', 'spam-analyzer'); ?></div>
                </div>
                <div class="stat-box stat-success">
                    <div class="stat-number"><?php echo number_format($stats['legitimate']); ?></div>
                    <div class="stat-label"><?php _e('Legitimate', 'spam-analyzer'); ?></div>
                </div>
                <div class="stat-box stat-warning">
                    <div class="stat-number"><?php echo number_format($stats['blocked']); ?></div>
                    <div class="stat-label"><?php _e('Blocked', 'spam-analyzer'); ?></div>
                </div>
            </div>
            <p class="stats-period"><?php _e('Last 30 days', 'spam-analyzer'); ?></p>
            <p><a href="<?php echo admin_url('admin.php?page=spam-analyzer'); ?>"><?php _e('View Details', 'spam-analyzer'); ?></a></p>
        </div>
        <style>
            .spam-analyzer-widget .stats-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px; margin-bottom: 15px; }
            .spam-analyzer-widget .stat-box { text-align: center; padding: 15px; background: #f9f9f9; border-radius: 4px; }
            .spam-analyzer-widget .stat-number { font-size: 28px; font-weight: bold; color: #0073aa; }
            .spam-analyzer-widget .stat-box.stat-danger .stat-number { color: #dc3232; }
            .spam-analyzer-widget .stat-box.stat-success .stat-number { color: #46b450; }
            .spam-analyzer-widget .stat-box.stat-warning .stat-number { color: #ffb900; }
            .spam-analyzer-widget .stat-label { font-size: 12px; color: #666; margin-top: 5px; }
            .spam-analyzer-widget .stats-period { text-align: center; color: #666; font-size: 12px; margin: 10px 0; }
        </style>
        <?php
    }

    /**
     * Add action links to plugin page
     */
    public function add_action_links($links)
    {
        $settings_link = '<a href="' . admin_url('admin.php?page=' . $this->plugin_name) . '">' . __('Settings', 'spam-analyzer') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}
