<?php
/**
 * WHMCS SpamKiller Module
 *
 * AI-powered spam detection for WHMCS contact forms, support tickets, client registration and checkout
 *
 * @package    SpamAnalyzer
 * @author     SpamKiller
 * @copyright  Copyright (c) 2025 SpamKiller
 * @license    MIT License
 * @version    1.1.1
 * @link       https://spamkiller.io
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

require_once __DIR__ . '/lib/SpamAnalyzerClient.php';

/**
 * Module configuration
 */
function spamanalyzer_config()
{
    return [
        'name' => 'SpamKiller',
        'description' => 'AI-powered spam detection for contact forms, support tickets, client registration and checkout. Integrates with SpamKiller API to automatically detect and block spam submissions.',
        'version' => '1.1.1',
        'author' => 'SpamKiller',
        'language' => 'english',
        'fields' => [
            'api_key' => [
                'FriendlyName' => 'API Key',
                'Type' => 'text',
                'Size' => '50',
                'Description' => 'Enter your SpamKiller API key (get it from <a href="https://spamkiller.io/dashboard" target="_blank">spamkiller.io</a>)',
                'Required' => true,
            ],
            'spam_threshold' => [
                'FriendlyName' => 'Spam Threshold',
                'Type' => 'text',
                'Size' => '5',
                'Default' => '70',
                'Description' => 'Spam score threshold (0-100). Submissions above this score will be blocked. Recommended: 70',
                'Required' => true,
            ],
            'enable_contact_form' => [
                'FriendlyName' => 'Enable for Contact Forms',
                'Type' => 'yesno',
                'Description' => 'Enable spam detection for contact form submissions',
                'Default' => 'yes',
            ],
            'enable_tickets' => [
                'FriendlyName' => 'Enable for Support Tickets',
                'Type' => 'yesno',
                'Description' => 'Enable spam detection for new support ticket submissions',
                'Default' => 'yes',
            ],
            'enable_client_registration' => [
                'FriendlyName' => 'Enable for Client Registration',
                'Type' => 'yesno',
                'Description' => 'Enable spam detection for new client registrations',
                'Default' => 'yes',
            ],
            'enable_checkout' => [
                'FriendlyName' => 'Enable for Checkout',
                'Type' => 'yesno',
                'Description' => 'Enable spam detection for order checkout (prevent spam orders)',
                'Default' => 'yes',
            ],
            'block_mode' => [
                'FriendlyName' => 'Block Mode',
                'Type' => 'dropdown',
                'Options' => [
                    'block' => 'Block spam submissions',
                    'flag' => 'Flag for admin review (tickets only)',
                ],
                'Default' => 'block',
                'Description' => 'Choose how to handle spam detections',
            ],
            'log_validations' => [
                'FriendlyName' => 'Log Validations',
                'Type' => 'yesno',
                'Description' => 'Log all validation results to Activity Log',
                'Default' => 'no',
            ],
        ]
    ];
}

/**
 * Activate module
 */
function spamanalyzer_activate()
{
    // Create custom field table for storing spam scores
    $query = "CREATE TABLE IF NOT EXISTS `mod_spamanalyzer_logs` (
        `id` INT(10) NOT NULL AUTO_INCREMENT,
        `type` VARCHAR(50) NOT NULL,
        `entity_id` INT(10) NULL,
        `is_spam` TINYINT(1) NOT NULL,
        `spam_score` DECIMAL(5,2) NOT NULL,
        `confidence` DECIMAL(5,2) NOT NULL,
        `ip_address` VARCHAR(45) NULL,
        `email` VARCHAR(255) NULL,
        `blocked` TINYINT(1) NOT NULL DEFAULT 0,
        `validation_data` TEXT NULL,
        `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        KEY `type` (`type`),
        KEY `entity_id` (`entity_id`),
        KEY `created_at` (`created_at`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    full_query($query);

    return [
        'status' => 'success',
        'description' => 'SpamKiller module activated successfully. Please configure your API key in the module settings.',
    ];
}

/**
 * Deactivate module
 */
function spamanalyzer_deactivate()
{
    return [
        'status' => 'success',
        'description' => 'SpamKiller module deactivated. Your settings and logs are preserved.',
    ];
}

/**
 * Admin area output
 */
function spamanalyzer_output($vars)
{
    $modulelink = $vars['modulelink'];
    $version = $vars['version'];
    $apiKey = $vars['api_key'];

    echo '<div class="alert alert-info">
        <h4><i class="fas fa-shield-alt"></i> SpamKiller v' . htmlspecialchars($version) . '</h4>
        <p>AI-powered spam detection is active and protecting your WHMCS installation.</p>
    </div>';

    // Test API connection
    if (!empty($apiKey)) {
        try {
            $client = new SpamAnalyzerClient($apiKey);
            $status = $client->getStatus();

            // ✅ FIXED: Updated to new API response structure (user instead of project)
            echo '<div class="panel panel-success">
                <div class="panel-heading">
                    <h3 class="panel-title"><i class="fas fa-check-circle"></i> API Connection Status</h3>
                </div>
                <div class="panel-body">
                    <div class="row">
                        <div class="col-sm-6">
                            <strong>Status:</strong> <span class="label label-success">Connected</span><br>
                            <strong>User:</strong> ' . htmlspecialchars($status['data']['user']['name']) . '<br>
                            <strong>Email:</strong> ' . htmlspecialchars($status['data']['user']['email']) . '<br>
                            <strong>Plan:</strong> ' . htmlspecialchars($status['data']['subscription']['plan']) . ' (' . htmlspecialchars($status['data']['subscription']['interval']) . ')<br>
                        </div>
                        <div class="col-sm-6">
                            <strong>Total Credits Available:</strong> ' . number_format($status['data']['credits']['total_available']) . '<br>
                            <strong>Monthly Subscription:</strong> ' . number_format($status['data']['credits']['monthly_subscription']) . '<br>
                            <strong>Purchased Balance:</strong> ' . number_format($status['data']['credits']['purchased_balance']) . '<br>
                            <strong>Rollover Credits:</strong> ' . number_format($status['data']['credits']['rollover']) . '<br>
                        </div>
                    </div>
                </div>
            </div>';

        } catch (Exception $e) {
            echo '<div class="alert alert-danger">
                <h4><i class="fas fa-exclamation-triangle"></i> API Connection Failed</h4>
                <p>' . htmlspecialchars($e->getMessage()) . '</p>
                <p>Please check your API key configuration.</p>
            </div>';
        }
    } else {
        echo '<div class="alert alert-warning">
            <h4><i class="fas fa-exclamation-triangle"></i> API Key Not Configured</h4>
            <p>Please configure your SpamKiller API key in the module settings to start protecting your WHMCS installation.</p>
            <p><a href="https://spamkiller.io/dashboard" target="_blank" class="btn btn-primary">Get API Key</a></p>
        </div>';
    }

    // Statistics
    $stats = spamanalyzer_get_statistics();

    echo '<div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title"><i class="fas fa-chart-bar"></i> Statistics (Last 30 Days)</h3>
        </div>
        <div class="panel-body">
            <div class="row text-center">
                <div class="col-sm-3">
                    <div class="stat-box">
                        <h3>' . number_format($stats['total']) . '</h3>
                        <p>Total Validations</p>
                    </div>
                </div>
                <div class="col-sm-3">
                    <div class="stat-box">
                        <h3 class="text-danger">' . number_format($stats['spam']) . '</h3>
                        <p>Spam Detected</p>
                    </div>
                </div>
                <div class="col-sm-3">
                    <div class="stat-box">
                        <h3 class="text-success">' . number_format($stats['legitimate']) . '</h3>
                        <p>Legitimate</p>
                    </div>
                </div>
                <div class="col-sm-3">
                    <div class="stat-box">
                        <h3 class="text-warning">' . number_format($stats['blocked']) . '</h3>
                        <p>Blocked</p>
                    </div>
                </div>
            </div>
        </div>
    </div>';

    // Recent detections
    $recentLogs = spamanalyzer_get_recent_logs(10);

    echo '<div class="panel panel-default">
        <div class="panel-heading">
            <h3 class="panel-title"><i class="fas fa-list"></i> Recent Detections</h3>
        </div>
        <div class="panel-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Type</th>
                            <th>Email</th>
                            <th>Spam Score</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>';

    if (empty($recentLogs)) {
        echo '<tr><td colspan="5" class="text-center">No validations yet</td></tr>';
    } else {
        foreach ($recentLogs as $log) {
            $statusLabel = $log['is_spam'] ? 'danger' : 'success';
            $statusText = $log['is_spam'] ? 'Spam' : 'Legitimate';
            $blockedBadge = $log['blocked'] ? ' <span class="label label-danger">Blocked</span>' : '';

            echo '<tr>
                <td>' . date('Y-m-d H:i', strtotime($log['created_at'])) . '</td>
                <td>' . htmlspecialchars($log['type']) . '</td>
                <td>' . htmlspecialchars($log['email']) . '</td>
                <td><span class="label label-' . $statusLabel . '">' . $log['spam_score'] . '%</span></td>
                <td>' . $statusText . $blockedBadge . '</td>
            </tr>';
        }
    }

    echo '      </tbody>
                </table>
            </div>
        </div>
    </div>';

    echo '<style>
        .stat-box { padding: 15px; }
        .stat-box h3 { font-size: 32px; margin: 0 0 5px 0; }
        .stat-box p { margin: 0; color: #666; }
    </style>';
}

/**
 * Get statistics for dashboard
 */
function spamanalyzer_get_statistics()
{
    $thirtyDaysAgo = date('Y-m-d H:i:s', strtotime('-30 days'));

    $result = full_query("SELECT
        COUNT(*) as total,
        SUM(CASE WHEN is_spam = 1 THEN 1 ELSE 0 END) as spam,
        SUM(CASE WHEN is_spam = 0 THEN 1 ELSE 0 END) as legitimate,
        SUM(CASE WHEN blocked = 1 THEN 1 ELSE 0 END) as blocked
        FROM mod_spamanalyzer_logs
        WHERE created_at >= '$thirtyDaysAgo'");

    $data = mysql_fetch_assoc($result);

    return [
        'total' => (int)$data['total'],
        'spam' => (int)$data['spam'],
        'legitimate' => (int)$data['legitimate'],
        'blocked' => (int)$data['blocked'],
    ];
}

/**
 * Get recent logs
 */
function spamanalyzer_get_recent_logs($limit = 10)
{
    $logs = [];
    $result = full_query("SELECT * FROM mod_spamanalyzer_logs ORDER BY created_at DESC LIMIT " . (int)$limit);

    while ($row = mysql_fetch_assoc($result)) {
        $logs[] = $row;
    }

    return $logs;
}
