<?php
/**
 * SpamKiller API Client
 *
 * Simple PHP client for communicating with the SpamKiller API
 *
 * @package    SpamAnalyzer
 * @version    1.1.1
 */

class SpamAnalyzerClient
{
    /**
     * API endpoint
     */
    private const API_BASE_URL = 'https://spamkiller.io/api/v1';

    /**
     * API key
     */
    private $apiKey;

    /**
     * Constructor
     *
     * @param string $apiKey The SpamKiller API key
     */
    public function __construct($apiKey)
    {
        $this->apiKey = $apiKey;
    }

    /**
     * Get subscription status
     *
     * @return array API response
     * @throws Exception
     */
    public function getStatus()
    {
        return $this->request('GET', '/status');
    }

    /**
     * Validate content for spam
     *
     * @param array $data Validation data (should include context, email, body)
     * @return array API response with validation results
     * @throws Exception
     */
    public function validate($data)
    {
        // Pass data directly to API without conversion
        // API expects: ['context' => '...', 'email' => '...', 'body' => ['field' => 'value', ...]]
        return $this->request('POST', '/validate', $data);
    }

    /**
     * Make HTTP request to API
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $data Request data
     * @return array API response
     * @throws Exception
     */
    private function request($method, $endpoint, $data = [])
    {
        $url = self::API_BASE_URL . $endpoint;

        $headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
            'User-Agent: WHMCS-SpamAnalyzer/1.0',
        ];

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            throw new Exception('API Request Failed: ' . $error);
        }

        $result = json_decode($response, true);

        if ($httpCode !== 200) {
            $errorMessage = $result['message'] ?? 'Unknown error';
            throw new Exception('API Error (' . $httpCode . '): ' . $errorMessage);
        }

        if (!isset($result['success']) || !$result['success']) {
            throw new Exception('API returned unsuccessful response');
        }

        return $result;
    }

    /**
     * Check if API key is valid
     *
     * @return bool
     */
    public function isValidApiKey()
    {
        try {
            $this->getStatus();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Format validation data for logging
     *
     * @param array $validationResult API validation result
     * @return string
     */
    public static function formatValidationForLog($validationResult)
    {
        if (!isset($validationResult['data'])) {
            return 'Invalid validation result';
        }

        $data = $validationResult['data'];
        $isSpam = $data['is_spam'] ? 'SPAM' : 'LEGITIMATE';
        $score = $data['spam_score'];
        $confidence = $data['confidence'];

        return sprintf(
            'Spam Detection: %s (Score: %.1f%%, Confidence: %.1f%%)',
            $isSpam,
            $score,
            $confidence
        );
    }
}
