<?php
/**
 * WHMCS SpamKiller Hooks
 *
 * Hooks to intercept and validate form submissions
 *
 * @package    SpamAnalyzer
 * @version    1.1.1
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

require_once __DIR__ . '/lib/SpamAnalyzerClient.php';

/**
 * Hook: Contact form submission
 */
add_hook('ContactFormSubmit', 1, function($vars) {
    $settings = spamanalyzer_get_settings();

    // Check if contact form validation is enabled
    if (!$settings['enable_contact_form']) {
        return;
    }

    // Prepare validation data
    $validationData = [
        'context' => 'Contact form submission for WHMCS hosting services',
        'email' => $vars['email'],
        'body' => [
            'name' => $vars['name'],
            'subject' => $vars['subject'],
            'message' => $vars['message'],
        ],
    ];

    // Validate with SpamKiller
    $result = spamanalyzer_validate($validationData, 'contact_form', null, $vars['email']);

    if (!$result) {
        return; // Error in validation, allow submission
    }

    // Check if spam and should be blocked
    if ($result['is_spam'] && $result['spam_score'] >= $settings['spam_threshold']) {
        if ($settings['block_mode'] === 'block') {
            // Block the submission
            return [
                'abort' => true,
                'message' => 'Your submission has been identified as spam and cannot be processed. If you believe this is an error, please contact us directly.',
            ];
        }
    }
});

/**
 * Hook: Support ticket validation (BEFORE creation)
 * ✅ FIXED: Using TicketOpenValidation instead of TicketOpen
 */
add_hook('TicketOpenValidation', 1, function($vars) {
    $settings = spamanalyzer_get_settings();

    // Check if ticket validation is enabled
    if (!$settings['enable_tickets']) {
        return;
    }

    // Get ticket data
    $userId = $vars['clientId'];
    $subject = $vars['subject'];
    $message = $vars['message'];
    $email = $vars['email'];

    // If email not provided, try to get from database
    if (!$email && $userId) {
        $client = Capsule::table('tblclients')
            ->where('id', $userId)
            ->first(['email']);
        $email = $client ? $client->email : null;
    }

    // If no email found, allow submission (can't validate without email)
    if (!$email) {
        return;
    }

    // Prepare validation data
    $validationData = [
        'context' => 'Support ticket for WHMCS hosting services - customer support inquiry',
        'email' => $email,
        'body' => [
            'subject' => $subject,
            'message' => $message,
            'department' => $vars['deptname'] ?? '',
        ],
    ];

    // Validate with SpamKiller (no ticketid yet - not created)
    $result = spamanalyzer_validate($validationData, 'ticket', null, $email);

    if (!$result) {
        return; // Error in validation, allow submission
    }

    // Check if spam and should be blocked
    if ($result['is_spam'] && $result['spam_score'] >= $settings['spam_threshold']) {
        if ($settings['block_mode'] === 'block') {
            // Block the ticket creation by returning error message
            return 'Your support ticket has been identified as spam. If you need assistance, please contact us through alternative means.';
        }
        // Note: 'flag' mode handled in TicketOpen hook (after ticket is created)
    }
});

/**
 * Hook: Support ticket opened (AFTER creation) - For flagging spam
 * Only used when block_mode = 'flag'
 */
add_hook('TicketOpen', 1, function($vars) {
    $settings = spamanalyzer_get_settings();

    // Only process if flag mode is enabled
    if (!$settings['enable_tickets'] || $settings['block_mode'] !== 'flag') {
        return;
    }

    // Get ticket data
    $userId = $vars['userid'];
    $subject = $vars['subject'];
    $message = $vars['message'];
    $email = null;

    // Get email from database
    if ($userId) {
        $client = Capsule::table('tblclients')
            ->where('id', $userId)
            ->first(['email']);
        $email = $client ? $client->email : null;
    }

    // If no email found, skip
    if (!$email) {
        return;
    }

    // Prepare validation data
    $validationData = [
        'context' => 'Support ticket for WHMCS hosting services - customer support inquiry',
        'email' => $email,
        'body' => [
            'subject' => $subject,
            'message' => $message,
            'department' => $vars['deptname'] ?? '',
        ],
    ];

    // Validate with SpamKiller
    $result = spamanalyzer_validate($validationData, 'ticket', $vars['ticketid'], $email);

    if (!$result) {
        return; // Error in validation, skip
    }

    // Flag ticket if spam detected
    if ($result['is_spam'] && $result['spam_score'] >= $settings['spam_threshold']) {
        // Flag ticket for admin review
        localAPI('UpdateTicket', [
            'ticketid' => $vars['ticketid'],
            'flag' => 1,
        ]);

        // Add admin note
        localAPI('AddTicketNote', [
            'ticketid' => $vars['ticketid'],
            'message' => sprintf(
                'SpamKiller flagged this ticket as potential spam (Score: %.1f%%, Confidence: %.1f%%)',
                $result['spam_score'],
                $result['confidence']
            ),
        ]);
    }
});

/**
 * Hook: Client registration validation (BEFORE creation)
 * ✅ FIXED: Using ClientDetailsValidation instead of ClientAdd
 */
add_hook('ClientDetailsValidation', 1, function($vars) {
    $settings = spamanalyzer_get_settings();

    // Check if client registration validation is enabled
    if (!$settings['enable_client_registration']) {
        return;
    }

    // Prepare validation data
    $validationData = [
        'context' => 'New client registration for WHMCS hosting services',
        'email' => $vars['email'],
        'body' => [
            'firstname' => $vars['firstname'] ?? '',
            'lastname' => $vars['lastname'] ?? '',
            'companyname' => $vars['companyname'] ?? '',
            'address1' => $vars['address1'] ?? '',
            'city' => $vars['city'] ?? '',
            'state' => $vars['state'] ?? '',
            'postcode' => $vars['postcode'] ?? '',
            'country' => $vars['country'] ?? '',
            'phonenumber' => $vars['phonenumber'] ?? '',
        ],
    ];

    // Validate with SpamKiller
    $result = spamanalyzer_validate($validationData, 'registration', null, $vars['email']);

    if (!$result) {
        return; // Error in validation, allow registration
    }

    // Check if spam and should be blocked
    if ($result['is_spam'] && $result['spam_score'] >= $settings['spam_threshold']) {
        if ($settings['block_mode'] === 'block') {
            // Return error message to prevent registration
            return sprintf(
                'Your registration has been identified as spam (Score: %.1f%%). If you believe this is an error, please contact us directly.',
                $result['spam_score']
            );
        }
    }

    // Allow registration
    return;
});

/**
 * Hook: Checkout validation (BEFORE order creation)
 * ✅ NEW: Added ShoppingCartValidateCheckout hook
 */
add_hook('ShoppingCartValidateCheckout', 1, function($vars) {
    $settings = spamanalyzer_get_settings();

    // Check if checkout validation is enabled
    if (!$settings['enable_checkout']) {
        return;
    }

    // Extract customer data
    $email = $vars['email'] ?? '';
    $firstname = $vars['firstname'] ?? '';
    $lastname = $vars['lastname'] ?? '';
    $companyname = $vars['companyname'] ?? '';
    $address1 = $vars['address1'] ?? '';
    $city = $vars['city'] ?? '';
    $state = $vars['state'] ?? '';
    $postcode = $vars['postcode'] ?? '';
    $country = $vars['country'] ?? '';
    $phonenumber = $vars['phonenumber'] ?? '';

    // Prepare validation data
    $validationData = [
        'context' => 'Order checkout for WHMCS hosting services - new order submission',
        'email' => $email,
        'body' => [
            'firstname' => $firstname,
            'lastname' => $lastname,
            'companyname' => $companyname,
            'address1' => $address1,
            'city' => $city,
            'state' => $state,
            'postcode' => $postcode,
            'country' => $country,
            'phonenumber' => $phonenumber,
            'paymentmethod' => $vars['paymentmethod'] ?? '',
        ],
    ];

    // Validate with SpamKiller
    $result = spamanalyzer_validate($validationData, 'checkout', null, $email);

    if (!$result) {
        return; // Error in validation, allow checkout
    }

    // Check if spam and should be blocked
    if ($result['is_spam'] && $result['spam_score'] >= $settings['spam_threshold']) {
        if ($settings['block_mode'] === 'block') {
            // Return error message(s) to prevent order creation
            return [
                sprintf(
                    'Your order has been identified as spam (Score: %.1f%%). If you believe this is an error, please contact us directly.',
                    $result['spam_score']
                ),
            ];
        }
    }

    // Allow checkout
    return;
});

/**
 * Get module settings
 */
function spamanalyzer_get_settings()
{
    $settings = Capsule::table('tbladdonmodules')
        ->where('module', 'spamanalyzer')
        ->pluck('value', 'setting')
        ->toArray();

    return [
        'api_key' => $settings['api_key'] ?? '',
        'spam_threshold' => (float)($settings['spam_threshold'] ?? 70),
        'enable_contact_form' => ($settings['enable_contact_form'] ?? 'on') === 'on',
        'enable_tickets' => ($settings['enable_tickets'] ?? 'on') === 'on',
        'enable_client_registration' => ($settings['enable_client_registration'] ?? 'on') === 'on',
        'enable_checkout' => ($settings['enable_checkout'] ?? 'on') === 'on',
        'block_mode' => $settings['block_mode'] ?? 'block',
        'log_validations' => ($settings['log_validations'] ?? '') === 'on',
    ];
}

/**
 * Validate content with SpamKiller API
 */
function spamanalyzer_validate($data, $type, $entityId, $email)
{
    $settings = spamanalyzer_get_settings();

    if (empty($settings['api_key'])) {
        return null; // No API key configured
    }

    try {
        $client = new SpamAnalyzerClient($settings['api_key']);
        $result = $client->validate($data);

        if (!isset($result['data'])) {
            throw new Exception('Invalid API response');
        }

        $validationData = $result['data'];

        // Log validation
        spamanalyzer_log_validation(
            $type,
            $entityId,
            $validationData['is_spam'],
            $validationData['spam_score'],
            $validationData['confidence'],
            $_SERVER['REMOTE_ADDR'] ?? null,
            $email,
            false, // Will be set to true if blocked
            json_encode($validationData)
        );

        // Log to activity log if enabled
        if ($settings['log_validations']) {
            $statusText = $validationData['is_spam'] ? 'SPAM' : 'LEGITIMATE';
            logActivity(sprintf(
                'SpamKiller: %s validation - %s (Score: %.1f%%, Confidence: %.1f%%) - Email: %s',
                ucfirst($type),
                $statusText,
                $validationData['spam_score'],
                $validationData['confidence'],
                $email
            ));
        }

        return $validationData;

    } catch (Exception $e) {
        // Log error
        logActivity('SpamKiller Error: ' . $e->getMessage());
        return null; // Allow submission on error
    }
}

/**
 * Log validation to database
 */
function spamanalyzer_log_validation($type, $entityId, $isSpam, $spamScore, $confidence, $ipAddress, $email, $blocked, $validationData)
{
    try {
        Capsule::table('mod_spamanalyzer_logs')->insert([
            'type' => $type,
            'entity_id' => $entityId,
            'is_spam' => $isSpam ? 1 : 0,
            'spam_score' => $spamScore,
            'confidence' => $confidence,
            'ip_address' => $ipAddress,
            'email' => $email,
            'blocked' => $blocked ? 1 : 0,
            'validation_data' => $validationData,
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    } catch (Exception $e) {
        logActivity('SpamKiller: Failed to log validation - ' . $e->getMessage());
    }
}

/**
 * Update blocked status for a log entry
 */
function spamanalyzer_update_blocked_status($type, $entityId)
{
    try {
        Capsule::table('mod_spamanalyzer_logs')
            ->where('type', $type)
            ->where('entity_id', $entityId)
            ->update(['blocked' => 1]);
    } catch (Exception $e) {
        logActivity('SpamKiller: Failed to update blocked status - ' . $e->getMessage());
    }
}
