<?php
/**
 * SpamKiller for PrestaShop
 *
 * AI-powered spam detection for PrestaShop contact forms, reviews, and registrations
 *
 * @author    SpamKiller
 * @copyright 2025 SpamKiller
 * @license   MIT License
 * @version   1.0.0
 * @link      https://spamkiller.io
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once _PS_MODULE_DIR_ . 'spamanalyzer/classes/SpamAnalyzerAPI.php';

class SpamAnalyzer extends Module
{
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->name = 'spamanalyzer';
        $this->tab = 'administration';
        $this->version = '1.0.0';
        $this->author = 'SpamKiller';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('SpamKiller');
        $this->description = $this->l('AI-powered spam detection for contact forms, product reviews, and customer registrations.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall? Your settings will be lost.');
    }

    /**
     * Install module
     */
    public function install()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        // Create database table for logs
        $sql = 'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'spamanalyzer_log` (
            `id_log` int(11) NOT NULL AUTO_INCREMENT,
            `type` varchar(50) NOT NULL,
            `entity_id` int(11) DEFAULT NULL,
            `is_spam` tinyint(1) NOT NULL,
            `spam_score` decimal(5,2) NOT NULL,
            `confidence` decimal(5,2) NOT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `email` varchar(255) DEFAULT NULL,
            `blocked` tinyint(1) NOT NULL DEFAULT 0,
            `validation_data` text,
            `date_add` datetime NOT NULL,
            PRIMARY KEY (`id_log`),
            KEY `type` (`type`),
            KEY `date_add` (`date_add`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;';

        if (!Db::getInstance()->execute($sql)) {
            return false;
        }

        // Set default configuration values
        Configuration::updateValue('SPAMANALYZER_API_KEY', '');
        Configuration::updateValue('SPAMANALYZER_THRESHOLD', 70);
        Configuration::updateValue('SPAMANALYZER_CONTACT_FORM', 1);
        Configuration::updateValue('SPAMANALYZER_CUSTOMER_REG', 1);
        Configuration::updateValue('SPAMANALYZER_PRODUCT_COMMENT', 1);
        Configuration::updateValue('SPAMANALYZER_BLOCK_MODE', 'block');
        Configuration::updateValue('SPAMANALYZER_LOG_ACTIVITY', 0);

        // Register hooks
        return parent::install()
            && $this->registerHook('actionContactFormSubmitBefore')
            && $this->registerHook('actionCustomerAccountAdd')
            && $this->registerHook('actionObjectCustomerMessageAddAfter')
            && $this->registerHook('actionProductCommentValidate')
            && $this->registerHook('displayBackOfficeHeader')
            && $this->registerHook('actionValidateCustomerAddressForm');
    }

    /**
     * Uninstall module
     */
    public function uninstall()
    {
        // Delete configuration
        Configuration::deleteByName('SPAMANALYZER_API_KEY');
        Configuration::deleteByName('SPAMANALYZER_THRESHOLD');
        Configuration::deleteByName('SPAMANALYZER_CONTACT_FORM');
        Configuration::deleteByName('SPAMANALYZER_CUSTOMER_REG');
        Configuration::deleteByName('SPAMANALYZER_PRODUCT_COMMENT');
        Configuration::deleteByName('SPAMANALYZER_BLOCK_MODE');
        Configuration::deleteByName('SPAMANALYZER_LOG_ACTIVITY');

        // Keep the database table for historical data
        // Uncomment to drop table on uninstall:
        // Db::getInstance()->execute('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'spamanalyzer_log`');

        return parent::uninstall();
    }

    /**
     * Module configuration page
     */
    public function getContent()
    {
        $output = '';

        // Process form submission
        if (Tools::isSubmit('submitSpamAnalyzer')) {
            $apiKey = Tools::getValue('SPAMANALYZER_API_KEY');
            $threshold = (int)Tools::getValue('SPAMANALYZER_THRESHOLD');

            // Validate inputs
            if (empty($apiKey)) {
                $output .= $this->displayError($this->l('API Key is required.'));
            } elseif ($threshold < 0 || $threshold > 100) {
                $output .= $this->displayError($this->l('Spam threshold must be between 0 and 100.'));
            } else {
                // Test API connection
                try {
                    $api = new SpamAnalyzerAPI($apiKey);
                    $status = $api->getStatus();

                    // Save configuration
                    Configuration::updateValue('SPAMANALYZER_API_KEY', $apiKey);
                    Configuration::updateValue('SPAMANALYZER_THRESHOLD', $threshold);
                    Configuration::updateValue('SPAMANALYZER_CONTACT_FORM', Tools::getValue('SPAMANALYZER_CONTACT_FORM'));
                    Configuration::updateValue('SPAMANALYZER_CUSTOMER_REG', Tools::getValue('SPAMANALYZER_CUSTOMER_REG'));
                    Configuration::updateValue('SPAMANALYZER_PRODUCT_COMMENT', Tools::getValue('SPAMANALYZER_PRODUCT_COMMENT'));
                    Configuration::updateValue('SPAMANALYZER_BLOCK_MODE', Tools::getValue('SPAMANALYZER_BLOCK_MODE'));
                    Configuration::updateValue('SPAMANALYZER_LOG_ACTIVITY', Tools::getValue('SPAMANALYZER_LOG_ACTIVITY'));

                    $output .= $this->displayConfirmation($this->l('Settings updated successfully!'));
                } catch (Exception $e) {
                    $output .= $this->displayError($this->l('API connection failed: ') . $e->getMessage());
                }
            }
        }

        // Display API status if configured
        $apiKey = Configuration::get('SPAMANALYZER_API_KEY');
        if (!empty($apiKey)) {
            $output .= $this->renderApiStatus($apiKey);
        }

        // Display statistics
        $output .= $this->renderStatistics();

        // Display configuration form
        $output .= $this->renderForm();

        return $output;
    }

    /**
     * Render API status box
     */
    private function renderApiStatus($apiKey)
    {
        try {
            $api = new SpamAnalyzerAPI($apiKey);
            $status = $api->getStatus();
            $data = $status['data'];

            $this->context->smarty->assign(array(
                'project_name' => $data['project']['name'],
                'plan_name' => $data['subscription']['plan']['name'],
                'validations_used' => $data['subscription']['usage']['validations_used'],
                'validations_remaining' => $data['subscription']['usage']['validations_remaining'],
                'validations_limit' => $data['subscription']['usage']['validations_limit'],
                'usage_percentage' => $data['subscription']['usage']['usage_percentage'],
            ));

            return $this->display(__FILE__, 'views/templates/admin/status.tpl');
        } catch (Exception $e) {
            return $this->displayError($this->l('API connection error: ') . $e->getMessage());
        }
    }

    /**
     * Render statistics
     */
    private function renderStatistics()
    {
        $thirtyDaysAgo = date('Y-m-d H:i:s', strtotime('-30 days'));

        $sql = 'SELECT
            COUNT(*) as total,
            SUM(CASE WHEN is_spam = 1 THEN 1 ELSE 0 END) as spam,
            SUM(CASE WHEN is_spam = 0 THEN 1 ELSE 0 END) as legitimate,
            SUM(CASE WHEN blocked = 1 THEN 1 ELSE 0 END) as blocked
            FROM `' . _DB_PREFIX_ . 'spamanalyzer_log`
            WHERE date_add >= "' . pSQL($thirtyDaysAgo) . '"';

        $stats = Db::getInstance()->getRow($sql);

        $this->context->smarty->assign(array(
            'stats_total' => (int)$stats['total'],
            'stats_spam' => (int)$stats['spam'],
            'stats_legitimate' => (int)$stats['legitimate'],
            'stats_blocked' => (int)$stats['blocked'],
        ));

        return $this->display(__FILE__, 'views/templates/admin/statistics.tpl');
    }

    /**
     * Render configuration form
     */
    private function renderForm()
    {
        $fields_form = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('SpamKiller Configuration'),
                    'icon' => 'icon-cogs'
                ),
                'input' => array(
                    array(
                        'type' => 'text',
                        'label' => $this->l('API Key'),
                        'name' => 'SPAMANALYZER_API_KEY',
                        'size' => 50,
                        'required' => true,
                        'desc' => $this->l('Enter your SpamKiller API key. Get it from') . ' <a href="https://spamkiller.io/dashboard" target="_blank">spamkiller.io</a>'
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Spam Threshold'),
                        'name' => 'SPAMANALYZER_THRESHOLD',
                        'size' => 5,
                        'required' => true,
                        'desc' => $this->l('Score above which content is considered spam (0-100). Recommended: 70')
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Enable for Contact Forms'),
                        'name' => 'SPAMANALYZER_CONTACT_FORM',
                        'is_bool' => true,
                        'desc' => $this->l('Validate contact form submissions'),
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Enabled')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('Disabled'))
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Enable for Customer Registration'),
                        'name' => 'SPAMANALYZER_CUSTOMER_REG',
                        'is_bool' => true,
                        'desc' => $this->l('Validate new customer registrations'),
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Enabled')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('Disabled'))
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Enable for Product Comments'),
                        'name' => 'SPAMANALYZER_PRODUCT_COMMENT',
                        'is_bool' => true,
                        'desc' => $this->l('Validate product reviews and comments'),
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Enabled')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('Disabled'))
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Block Mode'),
                        'name' => 'SPAMANALYZER_BLOCK_MODE',
                        'desc' => $this->l('Choose how to handle spam detections'),
                        'options' => array(
                            'query' => array(
                                array('id' => 'block', 'name' => $this->l('Block spam submissions')),
                                array('id' => 'log', 'name' => $this->l('Log only (allow but record)'))
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Log to PrestaShop Logs'),
                        'name' => 'SPAMANALYZER_LOG_ACTIVITY',
                        'is_bool' => true,
                        'desc' => $this->l('Log validations to PrestaShop activity log'),
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Enabled')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('Disabled'))
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right'
                )
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitSpamAnalyzer';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form));
    }

    /**
     * Get configuration field values
     */
    public function getConfigFieldsValues()
    {
        return array(
            'SPAMANALYZER_API_KEY' => Configuration::get('SPAMANALYZER_API_KEY'),
            'SPAMANALYZER_THRESHOLD' => Configuration::get('SPAMANALYZER_THRESHOLD'),
            'SPAMANALYZER_CONTACT_FORM' => Configuration::get('SPAMANALYZER_CONTACT_FORM'),
            'SPAMANALYZER_CUSTOMER_REG' => Configuration::get('SPAMANALYZER_CUSTOMER_REG'),
            'SPAMANALYZER_PRODUCT_COMMENT' => Configuration::get('SPAMANALYZER_PRODUCT_COMMENT'),
            'SPAMANALYZER_BLOCK_MODE' => Configuration::get('SPAMANALYZER_BLOCK_MODE'),
            'SPAMANALYZER_LOG_ACTIVITY' => Configuration::get('SPAMANALYZER_LOG_ACTIVITY'),
        );
    }

    /**
     * Hook: Before contact form submission (PS 1.7+)
     */
    public function hookActionContactFormSubmitBefore($params)
    {
        if (!Configuration::get('SPAMANALYZER_CONTACT_FORM')) {
            return;
        }

        $message = isset($params['message']) ? $params['message'] : '';
        $email = isset($params['from']) ? $params['from'] : '';

        $result = $this->validateContent(array(
            'context' => 'contact_form',
            'email' => $email,
            'message' => $message,
        ), 'contact_form', null, $email);

        if ($result && $this->shouldBlock($result)) {
            // Throw exception to stop form submission
            throw new PrestaShopException($this->l('Your message has been identified as spam and cannot be sent.'));
        }
    }

    /**
     * Hook: Customer registration
     */
    public function hookActionCustomerAccountAdd($params)
    {
        if (!Configuration::get('SPAMANALYZER_CUSTOMER_REG')) {
            return;
        }

        $customer = $params['newCustomer'];

        $result = $this->validateContent(array(
            'context' => 'registration',
            'email' => $customer->email,
            'body' => array(
                array('name' => 'firstname', 'value' => $customer->firstname),
                array('name' => 'lastname', 'value' => $customer->lastname),
            ),
        ), 'registration', $customer->id, $customer->email);

        if ($result && $this->shouldBlock($result)) {
            // Delete customer if spam
            $customer->delete();
            throw new PrestaShopException($this->l('Registration failed. Please contact support.'));
        }
    }

    /**
     * Hook: Product comment validation
     */
    public function hookActionProductCommentValidate($params)
    {
        if (!Configuration::get('SPAMANALYZER_PRODUCT_COMMENT')) {
            return;
        }

        if (!isset($params['object'])) {
            return;
        }

        $comment = $params['object'];

        $result = $this->validateContent(array(
            'context' => 'review',
            'message' => isset($comment->content) ? $comment->content : $comment->comment,
            'body' => array(
                array('name' => 'title', 'value' => isset($comment->title) ? $comment->title : ''),
            ),
        ), 'product_comment', $comment->id, '');

        if ($result && $this->shouldBlock($result)) {
            // Mark comment as not validated
            $comment->validate = 0;
            $comment->update();
        }
    }

    /**
     * Validate content with SpamKiller API
     */
    private function validateContent($data, $type, $entityId, $email)
    {
        $apiKey = Configuration::get('SPAMANALYZER_API_KEY');

        if (empty($apiKey)) {
            return null;
        }

        try {
            $api = new SpamAnalyzerAPI($apiKey);
            $result = $api->validate($data);

            if (!isset($result['data'])) {
                return null;
            }

            $validationData = $result['data'];

            // Log validation
            $this->logValidation(
                $type,
                $entityId,
                $validationData['is_spam'],
                $validationData['spam_score'],
                $validationData['confidence'],
                Tools::getRemoteAddr(),
                $email,
                false,
                json_encode($validationData)
            );

            // Log to PrestaShop logs if enabled
            if (Configuration::get('SPAMANALYZER_LOG_ACTIVITY')) {
                $statusText = $validationData['is_spam'] ? 'SPAM' : 'LEGITIMATE';
                PrestaShopLogger::addLog(
                    sprintf(
                        'SpamKiller: %s - %s (Score: %.1f%%, Confidence: %.1f%%)',
                        $type,
                        $statusText,
                        $validationData['spam_score'],
                        $validationData['confidence']
                    ),
                    1,
                    null,
                    'SpamAnalyzer'
                );
            }

            return $validationData;

        } catch (Exception $e) {
            PrestaShopLogger::addLog('SpamKiller Error: ' . $e->getMessage(), 3, null, 'SpamAnalyzer');
            return null;
        }
    }

    /**
     * Check if content should be blocked
     */
    private function shouldBlock($result)
    {
        $threshold = (float)Configuration::get('SPAMANALYZER_THRESHOLD');
        $blockMode = Configuration::get('SPAMANALYZER_BLOCK_MODE');

        return $result['is_spam']
            && $result['spam_score'] >= $threshold
            && $blockMode === 'block';
    }

    /**
     * Log validation to database
     */
    private function logValidation($type, $entityId, $isSpam, $spamScore, $confidence, $ipAddress, $email, $blocked, $validationData)
    {
        $sql = 'INSERT INTO `' . _DB_PREFIX_ . 'spamanalyzer_log`
            (type, entity_id, is_spam, spam_score, confidence, ip_address, email, blocked, validation_data, date_add)
            VALUES
            ("' . pSQL($type) . '", ' . (int)$entityId . ', ' . ($isSpam ? 1 : 0) . ',
            ' . (float)$spamScore . ', ' . (float)$confidence . ', "' . pSQL($ipAddress) . '",
            "' . pSQL($email) . '", ' . ($blocked ? 1 : 0) . ', "' . pSQL($validationData) . '", NOW())';

        Db::getInstance()->execute($sql);
    }
}
