<?php
/**
 * SpamKiller API Client
 *
 * @author    SpamKiller
 * @copyright 2025 SpamKiller
 * @license   MIT License
 * @version   1.0.0
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class SpamAnalyzerAPI
{
    /**
     * API endpoint
     */
    const API_BASE_URL = 'https://spamkiller.io/api/v1';

    /**
     * API key
     */
    private $apiKey;

    /**
     * Constructor
     *
     * @param string $apiKey
     */
    public function __construct($apiKey)
    {
        $this->apiKey = $apiKey;
    }

    /**
     * Get subscription status
     *
     * @return array
     * @throws Exception
     */
    public function getStatus()
    {
        return $this->request('GET', '/status');
    }

    /**
     * Validate content for spam
     *
     * @param array $data Must contain 'context', 'email' (required), and 'body' (object)
     * @return array
     * @throws Exception
     */
    public function validate($data)
    {
        // Email is required
        if (empty($data['email'])) {
            throw new Exception('Email is required for spam validation');
        }

        // Body must be provided
        if (empty($data['body']) || !is_array($data['body'])) {
            throw new Exception('Body must be a non-empty array');
        }

        // Body cannot contain 'email' key
        if (isset($data['body']['email'])) {
            throw new Exception('Body cannot contain "email" key. Email must be provided at top level.');
        }

        $payload = array(
            'context' => isset($data['context']) ? $data['context'] : 'Contact form submission',
            'email' => $data['email'],
            'body' => $data['body'],  // Body is now an object (associative array in PHP)
        );

        return $this->request('POST', '/validate', $payload);
    }

    /**
     * Make HTTP request to API
     *
     * @param string $method
     * @param string $endpoint
     * @param array $data
     * @return array
     * @throws Exception
     */
    private function request($method, $endpoint, $data = array())
    {
        $url = self::API_BASE_URL . $endpoint;

        $headers = array(
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
            'User-Agent: PrestaShop-SpamAnalyzer/1.0',
        );

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);

        curl_close($ch);

        if ($error) {
            throw new Exception('API Request Failed: ' . $error);
        }

        $result = json_decode($response, true);

        if ($httpCode !== 200) {
            $errorMessage = isset($result['message']) ? $result['message'] : 'Unknown error';
            throw new Exception('API Error (' . $httpCode . '): ' . $errorMessage);
        }

        if (!isset($result['success']) || !$result['success']) {
            throw new Exception('API returned unsuccessful response');
        }

        return $result;
    }

    /**
     * Check if API key is valid
     *
     * @return bool
     */
    public function isValidApiKey()
    {
        try {
            $this->getStatus();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}
